#include <stdio.h>
#include <windows.h>
#include <stdint.h>
#include "uart.h"



char port_name[20];
HANDLE hSerial;



/*
 * Close the current COM port connection 
 */
void
uart_close( void )
{
  CloseHandle(hSerial);
}



uint8_t
uart_setup( uint16_t cport, uint8_t rate )
{
  DCB dcb;
  COMMTIMEOUTS timeouts;

  /* create the COM port name from the int */
  sprintf(port_name, ((cport < 10) ? "COM%d" : "\\\\.\\COM%d"), cport);

  /* create a handle to connect to the port */
  hSerial = CreateFileA( port_name, GENERIC_READ | GENERIC_WRITE, 0,
                         NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );

  /* check to see if handle is valid */
  if( hSerial == INVALID_HANDLE_VALUE ) {

    if( GetLastError() == ERROR_FILE_NOT_FOUND ) {
      printf("error: port %s was not found\n", port_name);
      return 1;
    }

    printf("error: unable to connect to port %s\n", port_name);
    return 1;
  }

  /* set port parameters (baud, etc) */
  dcb.DCBlength = sizeof(DCB);
  if ( !GetCommState(hSerial, &dcb) ) {
    printf("error: unable to retrieve DCB for port: %s\n", port_name);
    return 1;
  }

  switch( rate ) {
    case 1: dcb.BaudRate = 9600; break;
    case 2: dcb.BaudRate = 115200; break;
    case 3: dcb.BaudRate = 460800; break;
  }

  /* disable XON/XOFF flow control */
  dcb.fBinary = TRUE;
  dcb.fParity = FALSE;
  dcb.fAbortOnError = FALSE;
  dcb.fNull = FALSE;
  dcb.fOutX = FALSE;

  dcb.ByteSize = 8;
  dcb.StopBits = ONESTOPBIT;
  dcb.Parity = NOPARITY;

  /* write DCB changes back to connection */
  if( !SetCommState(hSerial, &dcb) ) {
    printf("error: unable to write DCB changes for port: %s\n", port_name);
    return 1;
  }

  /* assign the timeout values for the port */
  timeouts.ReadIntervalTimeout = 50;
  timeouts.ReadTotalTimeoutConstant = 50;
  timeouts.ReadTotalTimeoutMultiplier = 10;
  timeouts.WriteTotalTimeoutConstant = 50;
  timeouts.WriteTotalTimeoutMultiplier = 10;

  if( !SetCommTimeouts(hSerial, &timeouts) ) {
    printf("error: unable to set timeout values for port: %s\n", port_name);
    return 1;
  }

  PurgeComm(hSerial, PURGE_TXCLEAR | PURGE_RXCLEAR);

  /* success */
  return 0;
}



/*
** Application code uses this to receive a byte
** RETURN: 0 - success
**         1 - failed
*/
uint8_t
uart_rx( uint8_t *val )
{
  DWORD bytesRead = 0;

  if( !ReadFile(hSerial, val, 1, &bytesRead, NULL) )
    return 1;

  if( bytesRead <= 0 )
    return 1;

  return 0;
}



void
uart_tx_buf( uint8_t *buf, uint16_t buf_len )
{
  DWORD bytesWritten = 0;
  WriteFile(hSerial, buf, buf_len, &bytesWritten, NULL);
  if( bytesWritten != buf_len ) {
    printf("error: buf write failed\n");
    fflush(stdout);
    uart_close();
    exit(1);
  }
}
