#include <stdio.h>
#include "stdint.h"
#include "vna.h"
#include "msgs.h"
#include "main.h"
#include "bits.h"



/* converts a msb first buf to unsigned 24 */
uint32_t
msbbufto24( uint8_t *buf )
{
  uint32_t ret;

  ret = buf[0];
  ret <<= 8;
  ret |= buf[1];
  ret <<= 8;
  ret |= buf[2];

  return ret;
}



/* converts a msb first buf to unsigned 32 */
uint32_t
msbbufto32( uint8_t *buf )
{
  uint32_t ret;

  ret = buf[0];
  ret <<= 8;
  ret |= buf[1];
  ret <<= 8;
  ret |= buf[2];
  ret <<= 8;
  ret |= buf[3];

  return ret;
}



/*
** Print out a J1587 message.
*/
void
msgs_rx_j1708( uint8_t *buf, uint16_t buf_len )
{
  uint16_t cnt;

  printf("j1708 rx: pid=%d ", buf[2] );
  printf("m=%01d ", buf[1]);
  printf("d[]=");
  for( cnt = 3; cnt < buf_len; cnt++ )
    printf("%03d ", buf[cnt]);
  printf("\n");
}



/*
** Print out a J1587 message.
*/
void
msgs_rx_j1587( uint8_t *buf, uint16_t buf_len )
{
  uint16_t cnt;

  printf("j1587 rx: pid=%d ", buf[3] );
  printf("m=%01d ", buf[1]);
  printf("d[]=");
  for( cnt = 4; cnt < buf_len; cnt++ )
    printf("%03d ", buf[cnt]);
  printf("\n");
}



/*
** Print out a J1939 message.
*/
void
msgs_rx_j1939( uint8_t *buf, uint16_t buf_len )
{
  uint16_t cnt;

  printf("rx: pgn=%d ", msbbufto24(&buf[2]));
  printf("p=%01d ", buf[1]);
  printf("d=%03d ", buf[5]);
  printf("s=%03d ", buf[6]);
  printf("pri=%03d ", buf[7]);
  printf("d[]=");
  for( cnt = 8; cnt < buf_len; cnt++ )
    printf("%03d ", buf[cnt]);
  printf("\n");
}



/*
** Print out a raw CAN frame.
*/
void
msgs_rx_can( uint8_t *buf, uint16_t buf_len )
{
  uint16_t cnt;
  uint32_t id;

  id = msbbufto32(&buf[2]);

  if( id & B31 )
    printf("rx: id-e=%lu ", id & ~B31);
  else
    printf("rx: id-s=%lu ", id);

  printf("d[]=");
  for( cnt = 6; cnt < buf_len; cnt++ )
    printf("%03d ", buf[cnt]);
  printf("\n");
}



void
msgs_gps( uint8_t *buf, uint16_t buf_len )
{
  uint16_t cnt;
  double lat_min;
  double lon_min;

  lat_min = msbbufto24(&buf[3]);
  lat_min /= 100000;
  lat_min += buf[2];

  lon_min = msbbufto24(&buf[9]);
  lon_min /= 100000;
  lon_min += buf[8];

  printf("Lat: %02d %06f %01c   Lon: %03d %06f %01c   SVs: %02d\n",
          buf[1], lat_min, buf[6], buf[7], lon_min, buf[12], buf[13]);

  printf("\n");
}





/*
** Print out a J1939 message.
*/
void
msgs_rx_i15765( uint8_t *buf, uint16_t buf_len )
{
  uint16_t cnt;

  printf("p=%01d ", buf[1]);
  printf("d=%03d ", buf[2]);
  printf("s=%03d ", buf[3]);
  printf("pri=%03d ", buf[4]);
  printf("tat=%03d ", buf[5]);
  printf("d[]=");
  for( cnt = 6; cnt < buf_len; cnt++ )
    printf("%03d ", buf[cnt]);
  printf("\n");
}





/*
** Print out a CAN message.
*/
void
msgs_stats( uint8_t *buf, uint16_t buf_len )
{
  static uint32_t statcnt = 0;
  printf("rx: j1708 valid msgs=%d ", msbbufto32(&buf[1]) );
  printf("j1708 invalid bytes=%d ", msbbufto32(&buf[5]) );
  printf("can frames=%d ", msbbufto32(&buf[9]) );
  printf("hwv=%d ", buf[13] );
  printf("swv=%d ", buf[14] );
  printf("#=%d ", ++statcnt );
  printf("\n");
}



/*
** Print out a CAN message.
*/
void
msgs_stats_obd( uint8_t *buf, uint16_t buf_len )
{
  printf("rx: can frames=%d ", msbbufto32(&buf[1]) );
  printf("hwv=%d ", buf[5] );
  printf("swv=%d ", buf[6] );
  printf("\n");
}



/*
** Print out the digital odometer.
*/
void
msgs_odometer( uint8_t *buf, uint16_t buf_len )
{
  printf("rx: digital odometer=%d ", msbbufto32(&buf[1]) );
  printf("\n");
}


/*
** Print out a CAN message.
*/
void
msgs_bl_ver( uint8_t *buf, uint16_t buf_len )
{
  printf("blv=%d ", buf[1] );
  printf("\n");
}

/*
** Print out a CAN message.
*/
void
msgs_vna_id( uint8_t *buf, uint16_t buf_len )
{
  printf("blv=%d ", buf[1] );
  printf("bl_hwv=%d ", buf[2] );
  printf("app_hwv=%d ", buf[3] );
  printf("app_swv=%d ", buf[4] );
  printf("\n");
}


/*
** Print out BT MAC address
*/
void
msgs_mac ( uint8_t *buf, uint16_t buf_len )
{
  printf("MAC: %X:%X:%X:%X:%X:%X\n", 
    buf[1], buf[2], buf[3], buf[4], buf[5], buf[6] );
}


/*
** This function receives all completed messages from the VNA-232
*/
void
msgs_post( uint8_t *buf, uint16_t buf_len )
{
  static uint16_t ack_cnt = 0;

  /* process new msg */
  switch( buf[0] ) {

    /* received an ack */
    case VNA_MSG_ACK: {
      printf("rx: ack = %d\n", buf[1] );
      break;
    }

    /* received a j1587 */
    case VNA_MSG_RX_J1708: {
      msgs_rx_j1708( buf, buf_len );
      break;
    }

    /* received a j1587 */
    case VNA_MSG_RX_J1587: {
      msgs_rx_j1587( buf, buf_len );
      break;
    }

    /* received a j1939 */
    case VNA_MSG_RX_J1939: {
      msgs_rx_j1939( buf, buf_len );
      break;
    }

    /* received a j1939 */
    case VNA_MSG_RX_CAN: {
      msgs_rx_can( buf, buf_len );
      break;
    }

    /* received a iso 15765 message */
    case VNA_MSG_RX_I15765: {
      msgs_rx_i15765( buf, buf_len );
      break;
    }

    /* received a stat message */
    case VNA_MSG_STATS: {
      msgs_stats( buf, buf_len );
      break;
    }

    /* received a stat message */
    case VNA_MSG_STATS_OBD: {
      msgs_stats_obd( buf, buf_len );
      break;
    }

    case VNA_MSG_BLVER: {
      msgs_bl_ver( buf, buf_len );
      break;
    }

    case VNA_MSG_ODOMETER: {
      msgs_odometer( buf, buf_len );
      break;
    }

    case VNA_MSG_VNA_ID: {
      msgs_vna_id( buf, buf_len );
      break;
    }

    case VNA_MSG_GPS: {
      msgs_gps( buf, buf_len );
      break;
    }

    case VNA_MSG_MAC: {
      msgs_mac( buf, buf_len );
      break;
    }

    default: {
      printf("error: unknown message\n");
      break;
    }
  }

  fflush(stdout);
}
