#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <conio.h>
#include <stdint.h>
#include "rs232.h"
#include "uart.h"
#include "vna.h"
#include "bits.h"



void
main_tx_sleep ( void )
{
  uint8_t buf[6];
  
  buf[0] = VNA_MSG_SLEEP;  // command sleep state
  buf[1] = 0x37;  // key
  buf[2] = 0x55;  // key
  buf[3] = 0x00;  // key
  buf[4] = 0x73;  // key

  /* send to vna-232 */
  rs232_tx(buf,5);
  printf("info: command sleep mode\n");
}



void
main_tx_wake ( void )
{
  uint8_t buf[6];
  
  buf[0] = VNA_MSG_WAKE; // command wake state
  buf[1] = 0x15;  // key
  buf[2] = 0x00;  // key
  buf[3] = 0x55;  // key
  buf[4] = 0x51;  // key

  /* send to vna-232 */
  rs232_tx(buf,5);
  printf("info: command wake mode\n");
}


void
main_rs232_enable ( void )
{
  uint8_t buf[6];
  
  buf[0] = VNA_MSG_RS232_ENABLE;

  /* send to vna-232 */
  rs232_tx(buf,1);
  printf("info: turning on rs232\n");
}



void
main_tx_passall ( void )
{
  uint8_t buf[6];
  
  buf[0] = VNA_MSG_PAMODE_SET;
  buf[1] = 0;  // port
  buf[2] = 1;  // j1708
  buf[3] = 1;  // j1587
  buf[4] = 0;  // can
  buf[5] = 1;  // j1939

  /* send to vna-232 */
  rs232_tx(buf,6);
  printf("info: send pass all mode\n");
}


void
main_tx_odometer ( void )
{
  uint8_t buf[6];
  
  /* set the odometer */
  buf[0] = VNA_MSG_ODOMETER;
  buf[1] = 0;
  buf[2] = 0;
  buf[3] = 0;
  buf[4] = 0;
  rs232_tx(buf,5);
  printf("info: setting odometer\n");
}


/*
** hard coded reset command (send it twice)
*/
void
main_tx_reset ( void )
{
  uint8_t buf[6];
  buf[0] = 17;
  buf[1] = 90;
  buf[2] = 105;
  buf[3] = 165;
  rs232_tx(buf,4);
}


/*
** Transmit request for PGN.  Used for request based PGNs like VIN.
** dst: Destination address (255=global)
** pgn: Request PGN
*/
void
main_j1939_tx_request ( uint8_t dst, uint32_t pgn )
{
  uint8_t buf[50];
  
  buf[0] = VNA_MSG_TX_J1939;
  buf[1] = 0;      // port
  buf[2] = 0;      // pgn 23-16
  buf[3] = 0xEA;   // pgn 15 - 8
  buf[4] = 0;      // pgn 7 - 0
  buf[5] = 255;    // dst
  buf[6] = 43;     // src
  buf[7] = 6;      // pri
  buf[8] = (uint8_t) pgn;
  buf[9] = (uint8_t) (pgn >> 8);
  buf[10] = (uint8_t) (pgn >> 16);

  rs232_tx(buf,11);
  printf("info: transmitted request\n");
}



/*
** Transmit J1587 destination specific request for PID.
** dst: Destination address (must be less than 255)
** pid: PID being requested
*/
void
main_j1587_tx_dst_specific_request ( uint8_t dst, uint8_t pid )
{
  uint8_t buf[20];
  
  buf[0] = VNA_MSG_TX_J1587;
  buf[1] = 145;   // mid
  buf[2] = 0;     // pid 15 - 8 
  buf[3] = 128;   // pid 7 - 0 (pid 128 = dst req)
  buf[4] = 3;     // pri
  buf[5] = pid;
  buf[6] = dst;

  rs232_tx(buf,7);
  printf("info: transmitting destination specific request (pid=128)\n");
}



/*
** Transmit J1587 global request for PID.
** pid: PID being requested
*/
void
main_j1587_tx_global_request ( uint8_t pid )
{
  uint8_t buf[20];
  
  buf[0] = VNA_MSG_TX_J1587;
  buf[1] = 145;   // mid
  buf[2] = 0;     // pid 15 - 8 
  buf[3] = 0;     // pid 7 - 0 (pid 0 = global req)
  buf[4] = 3;     // pri
  buf[5] = pid;

  rs232_tx(buf,6);
  printf("info: transmitting global request (pid=0)\n");
}





/*
** Transmit BAM message (i.e. multi-frame to all controllers)
*/
void
main_j1939_tx_bam ( uint8_t p )
{
  uint8_t cnt;
  uint8_t buf[300];
  
  buf[0] = VNA_MSG_TX_J1939;
  buf[1] = p;      // port
  buf[2] = 0;      // pgn 23-16
  buf[3] = 0xFE;   // pgn 15 - 8
  buf[4] = 0xCA;   // pgn 7 - 0
  buf[5] = 255;    // dst
  buf[6] = 252;    // src
  buf[7] = 4;      // pri

  /* set to 'not availble' for since this is a test msg */
  for( cnt = 0; cnt < 250; cnt++ )
    buf[8+cnt] = cnt;

  rs232_tx( buf, 80+8 );
  printf("info: transmitted BAM of pgn=65226\n");
}



/*
** Transmit Power Takeoff Information msg (PGN 65264)
*/
void
main_j1939_tx ( uint8_t p, uint32_t pgn, uint8_t src )
{
  uint16_t cnt;
  uint8_t buf[800];
  
  buf[0] = VNA_MSG_TX_J1939;
  buf[1] = p;                      // port
  buf[2] = (uint8_t) (pgn >> 16);  // pgn 23-16
  buf[3] = (uint8_t) (pgn >> 8);   // pgn 15 - 8
  buf[4] = (uint8_t) pgn;          // pgn 7 - 0
  buf[5] = 255;                     // dst
  buf[6] = src;                    // src
  buf[7] = 5;                      // pri

  /* set to 'not availble' for since this is a test msg */
#if 0
  for( cnt = 0; cnt < 8; cnt++ )
    buf[8+cnt] = cnt;
#endif

  for( cnt = 0; cnt < 500; cnt++ )
    buf[8+cnt] = cnt;

  //rs232_tx(buf,16);
  rs232_tx(buf,8+270);
  //rs232_tx(buf,8+3);
  printf("info: transmitted power takeoff information onto j1939. (pgn=65264)\n");
}



/*
** Transmit J1587 message.
*/
void
main_j1587_tx ( uint16_t pid )
{
  uint8_t i = 0;
  uint8_t buf[30];
  
  buf[i++] = VNA_MSG_TX_J1587;
  buf[i++] = 145;   // mid
  buf[i++] = (uint8_t) (pid >> 8); // pid 15 - 8 
  buf[i++] = (uint8_t) pid;        // pid 7 - 0
  buf[i++] = 3;     // pri
  buf[i++] = 105;   // data 0
  buf[i++] = 171;   // data 0

  rs232_tx(buf,i);
  printf("info: transmitted j1587 pid = %d\n", pid);
}



/*
** Add filter for PID 133 (Average fuel rage)
*/
void
main_filter_add_j1587 ( uint16_t pid )
{
  uint8_t buf[5];
  
  /* add filter for cruise control/vehicle speed PGN */
  buf[0] = VNA_MSG_FA_J1587;
  buf[1] = pid >> 8;          // pid 15 - 8
  buf[2] = (uint8_t) pid;     // pid 7 - 0

  /* send to vna-232 */
  rs232_tx(buf,3);

  printf("info: added filter for j1587 pid: %d\n", pid);
}



/*
** Add filter for PGN 65265 (Cruise Control/Vehicle Speed)
*/
void
main_filter_add_j1939 ( uint32_t pgn )
{
  uint8_t buf[5];
  
  /* add filter for cruise control/vehicle speed PGN */
  buf[0] = VNA_MSG_FA_J1939;
  buf[1] = 0;                     // port 0
  buf[2] = (uint8_t)(pgn >> 16);  // pgn 23-16
  buf[3] = (uint8_t)(pgn >> 8);   // pgn 15 - 8
  buf[4] = (uint8_t)pgn;          // pgn 7 - 0

  /* send to vna-232 */
  rs232_tx(buf,5);
  printf("info: added filter for j1939 pgn: %d\n", pgn);
}



/*
** Add low pass filter for a PGN.
*/
void
main_filter_add_ext_j1939 ( uint32_t pgn )
{
  uint8_t buf[20];
  
  /* add low pass filter */
  buf[0] = VNA_MSG_FA_EXT_J1939;  // 50
  buf[1] = 0;                     // port 0
  buf[2] = (uint8_t)(pgn >> 16);  // pgn 23-16
  buf[3] = (uint8_t)(pgn >> 8);   // pgn 15 - 8
  buf[4] = (uint8_t)pgn;          // pgn 7 - 0

  /* send 32-bit time in milliseconds (e.g. every 100ms) */
  buf[5] = 0;
  buf[6] = 250;

  /* send to vna-232 */
  rs232_tx(buf,7);

  printf("info: added low pass filter for j1939 pgn @ 250ms: %d\n", pgn );
}



void
main_mconn_ext ( uint32_t type, uint32_t baud )
{
  uint8_t buf[10];
  buf[0] = VNA_MSG_MCONN_EXT;
  buf[1] = 0;
  buf[2] = type >> 24;
  buf[3] = type >> 16;
  buf[4] = type >> 8;
  buf[5] = type;
  buf[6] = baud >> 24;
  buf[7] = baud >> 16;
  buf[8] = baud >> 8;
  buf[9] = baud;
  rs232_tx ( buf, 10 );
}

void
main_req_mac ( void )
{
  uint8_t buf[3];

  buf[0] = VNA_MSG_REQ;
  buf[1] = VNA_MSG_MAC;
  rs232_tx(buf,2);
  printf("info: requesting bt mac address\n");
}



void
main ( int argc, char *argv[] )
{
  uint8_t val;
  uint8_t cnt;
  uint8_t rate;
  uint16_t port;
  static uint16_t time = 0;

  /* must be port and speed, where speed can only be 1, 2, or 3 */
  if( (argc != 3) || (argv[1][0] < '0') || (argv[1][0] > '9')
                  || (argv[2][0] < '1') || (argv[2][0] > '3') ) {
    printf("usage: pc232 <port> <rate>\n");
    printf("\n");
    printf("  ver. 1.1\n");
    printf("\n");
    printf("  <port> - serial port number\n");
    printf("  <speed> - 1=9.6K, 2=115.2K, 3=460.8K\n");
    return;
  }

  /* get port */
  port = strtol(argv[1], 0, 10);

  /* get data rate */
  rate = argv[2][0] - '0';

  /* if setup fails, do nothing */
  if( uart_setup(port,rate) )
    return;

  /* wait for RS-232 transceiver to stabilize */
  Sleep(100);

  /* add filter j1587 engine load */
  main_filter_add_j1587(92);

  /* add filter j1587 VIN */
  main_filter_add_j1587(237);

  /* send j1587 request for VIN from engine (0 = engine) */
  main_j1587_tx_dst_specific_request(0,237);

  /* send j1587 request for VIN to everyone */
  main_j1587_tx_global_request(237);

  /* added filter for engine speed PGN */
  main_filter_add_ext_j1939(61444);

  /* added filters for both odometer messages */
  main_filter_add_j1939(65217);
  main_filter_add_j1939(65248);

  /* add filter for VIN and transmit global request for VIN */
  main_filter_add_j1939(65260);
  main_j1939_tx_request(255,65260);

  /* wait for user to hit q */
  while( _kbhit() == 0 || toupper(getch()) != 'Q' ) {

    while( uart_rx(&val) == 0 )
      rs232_update(val);

    /* ping bluetooth devices to keep bluetooth awake */
    if( time++ == 20 ) {
      time = 0;
      main_filter_add_j1939(65248);
    }

    Sleep(10);
  }

  Sleep(1000);
  uart_close();
}
